/**********************************************************************
 *    Copyright (C) 1998 Anders Johansson <andersjo@lysator.liu.se>
 *    Copyright (C) 2001 Martin Renold <martinxyz@gmx.ch> (parport)
 *    Copyright (C) 2002 Jrg Mensmann <joerg.mensmann@gmx.de> 
 *                         (removed threading)
 * 
 * This  program is free software; you can redistribute it and/or modify it
 * under the  terms  of  the GNU General Public License as published by the
 * Free Software Foundation version 2 of the License.
 * 
 * This  program  is  distributed  in  the hope that it will be useful, but
 * WITHOUT   ANY   WARRANTY;   without   even  the   implied   warranty  of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details (see enclosed file COPYING).
 * 
 * You  should have received a copy of the GNU General Public License along
 * with this  program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 ***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <linux/kd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <assert.h>

/* Parport stuff */
#include <sys/io.h>
/* Hmmm... FIXME, hardcoded... but I don't care as long as it works ;) */
#define DATA 0x378
#define STATUS (DATA+1)
#define CONTROL (DATA+2)

#define NUMLOCKLED      1
#define CAPSLOCKLED     2
#define SCROLLLOCKLED   3
#define KEYBOARDDEVICE	"/dev/console"

typedef enum { CLEAR = 0, SET = 1, TOGGLE = 2 } LedMode;

/* Declarations of functions local to this file */
void led( int what, LedMode mode );
void toggle_led( int what );
void clear_led( int what ) { led( what+1, CLEAR ); }
void set_led( int what ) { led( what+1, SET ); }


static int	keyboardDevice = 0;

/* Which led and how many times to blink it */
int lednum;
int blinks = 0;
/* The non-blinking state of the led */
int led_on = 0;

extern int verbosity;
extern void message( int level, const char *format, ... );

void led_exit() {
    message( 3, "led_exit() called\n" );

    clear_led(lednum);

    if( keyboardDevice ) {	/* EUID root - CONSOLE */
	(void) close( keyboardDevice );
    }
}

void *led_loop() {
/* loops forever, blinking the led according to 
 * the blinks variable */
    int i;
    int laststate = 0;

    message( 3, "led_loop() started\n" );

    if( -1 == (keyboardDevice = open(KEYBOARDDEVICE, O_RDONLY))) {
	assert(0);
    }

    if (lednum >= 3) {
      if (ioperm(DATA,3,1)) {
	printf("Sorry, you were not able to gain access to the ports\n");
	printf("You must be root to run this program with parallel port support\n");
	exit(1);
      }      
    }

    while(1) {
      /* toggle led state? */
      if (led_on != laststate) {       
	laststate = led_on;
	if (laststate) set_led(lednum); else clear_led(lednum);
        usleep(100000);  /* sleep a little while */
      }

      /* blink */
      for( i = 0; i < blinks; i++ ) {
        toggle_led(lednum);
        message( 4, "o\b" );
        usleep(100000);
        toggle_led(lednum);
        message( 4, ".\b" );
        usleep(150000);
      }

      usleep(500000);
    }
}

void led( int led, LedMode mode ) {
/* This is the actual function that sets/clears/toggles an led */
    char ledVal;

    if ( led <= 3 ) {
      /* Keyboard LED */

      if( ioctl( keyboardDevice, KDGETLED, &ledVal )) {
	assert( 0 );
      }
      
      switch( led ) {
      case SCROLLLOCKLED:
	if( mode == SET )
	  ledVal |= LED_SCR;
	else if( mode == CLEAR )
	  ledVal &= ~LED_SCR;
	else
	  ledVal ^= LED_SCR;
	break;
      case NUMLOCKLED:
	if( mode == SET )
	  ledVal |= LED_NUM;
	else if( mode == CLEAR )
	  ledVal &= ~LED_NUM;
	else
	  ledVal ^= LED_NUM;
	break;
      case CAPSLOCKLED:
	if( mode == SET )
	  ledVal |= LED_CAP;
	else if( mode == CLEAR )
	  ledVal &= ~LED_CAP;
	else
	  ledVal ^= LED_CAP;
	break;
      default:
	assert( 0 );
      }

      if( ioctl( keyboardDevice, KDSETLED, ledVal )) {
	assert( 0 );
      }
    } else {
      /* Parallel port LED */
      led -= 4;
      assert(led >= 0 && led < 8);
      ledVal = inb(DATA);
      if( mode == SET )
	ledVal |= 1 << led;
      else if( mode == CLEAR )
	ledVal &= ~(1 << led);
      else
	ledVal ^= (1 << led);
      outb(ledVal, DATA);
    }
}

void toggle_led( int what ) {
    sigset_t sigset;

    /* block signals while we adjust the led */
    if( sigemptyset(&sigset)) assert(0);
    if( sigaddset(&sigset, SIGINT)) assert(0);
    if( sigaddset(&sigset, SIGTERM)) assert(0);
    if( sigprocmask(SIG_BLOCK, &sigset, NULL)) assert(0);

    led( what+1, TOGGLE );

    /* deliver any pending signals */
    if( sigprocmask(SIG_UNBLOCK, &sigset, NULL)) assert(0);
}
