/* lcd.c
 *
 * $Id: lcd.c,v 1.11 2004/03/08 03:55:11 mikey Exp $
 *
 * LCD driver for HD44780 compatible displays connected to the parallel port,
 * because real men use device files.
 *
 * Copyright (C) by Michael McLellan (mikey@mclellan.org.nz)
 * Released under the terms of the GNU GPL, see file COPYING for more details.
 */

#define KBUILD_MODNAME lcd


#include <linux/config.h>
#include <linux/version.h>
#ifndef KERNEL_VERSION
#define KERNEL_VERSION(a,b,c) (((a) << 16) + ((b) << 8) + (c))
#endif
#ifndef LINUX_VERSION_CODE
#define LINUX_VERSION_CODE KERNEL_VERSION(2,2,0)
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,0)
#include <asm/segment.h>
#include <linux/sched.h>
#else
#include <asm/uaccess.h>
#endif
#if CONFIG_PARPORT
#include <linux/parport.h>
#elif CONFIG_PARPORT_MODULE
#include <linux/parport.h>
#warning - Your kernel has parallel-port support compiled as a
#warning - module, you must ensure that the parport module is
#warning - inserted before lcdmod. This should happen automatically
#warning - whenever you send data to /dev/lcd if you "make install"
#else
#warning - Your kernel is not configured with parport support,
#warning - compiling with failsafe outb() functions instead.
#endif // PARPORT_CONFIG
#include <linux/proc_fs.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <asm/io.h>
#include "config.h"
#include "charmap.h"
#include "wiring.h"

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
#include <linux/cdev.h>
#endif

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,0)
/* does MODULE_LICENSE exist in older kernel trees? */
MODULE_LICENSE("GPL");
#endif

#ifdef CGRAM_DEFAULT
#include "cgram/default.h"
#endif
#ifdef CGRAM_SWEDISH
#include "cgram/swedish.h"
#endif

#define MAX_DISP_ROWS	4		// The HD44780 supports up to 4 rows
#define MAX_DISP_COLS	40		// The HD44780 supports up to 40 columns

/* input_state states */
#define NORMAL			0
#define ESC				1   	// Escape sequence start
#define DCA_Y			2   	// Direct cursor access, the next input will be the row
#define DCA_X			3   	// Direct cursor access, the next input will be the column
#define CGRAM_SELECT	4		// Selecting which slot to enter new character
#define CGRAM_ENTRY		5		// Adding a new character to the CGRAM
#define CGRAM_GENERATE	6		// Waiting fot the 8 bytes which define a character
#define CHAR_MAP_OLD	7		// Waiting for the original char to map to another
#define CHAR_MAP_NEW	8		// Waiting for the new char to replace the old one with
#define ESC_			10		// Waiting for the [ in escape sequence

/* globals */
#if CONFIG_PARPORT || CONFIG_PARPORT_MODULE
static struct pardevice *pd;
#endif
static int io = DFLT_BASE;
static int disp_rows = DFLT_DISP_ROWS;
static int disp_cols = DFLT_DISP_COLS;
static unsigned char state[ MAX_DISP_ROWS ][ MAX_DISP_COLS ];	// The current state of the display
static int disp_row = 0, disp_column = 0; 						// Current actual cursor position
static int row = 0, column = 0; 								// Current virtual cursor position
static int wrap = 0;											// Linewrap on/off
static char backlight = BL_ON;									// Backlight on-off

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
static struct cdev *my_cdev; // character device
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,1,0)
MODULE_DESCRIPTION( "LCD parallel port driver for HD44780 compatable controllers" );
MODULE_AUTHOR( "Michael McLellan <mikey@mclellan.org.nz>" );
MODULE_PARM( io, "i" );
MODULE_PARM( disp_rows, "i" );
MODULE_PARM( disp_cols, "i" );
MODULE_PARM_DESC( io, "Parallel port i/o base address (default: DEFAULT_BASE)" );
MODULE_PARM_DESC( disp_rows, "LCD rows (default: DFLT_DISP_ROWS, max: MAX_DISP_ROWS)" );
MODULE_PARM_DESC( disp_cols, "LCD columns (default: DFLT_DISP_COLS, max: MAX_DISP_COLS)" );
#endif


/* Send an instruction to the display, e.g. move cursor */
static void writeCommand( char command, int display )
{
	/* The E cycle time for a write should be at least 500 ns,
	 * and the minimum E pulse width should be 230 ns.
	 */
#if !CONFIG_PARPORT && !CONFIG_PARPORT_MODULE
	outb( (display==1?E1_ON:E2_ON) | RS_INST | RW_WRITE | backlight, io + 2 );
    udelay( 250 );
    outb( command, io );
    outb( (display==1?E1_OFF:E2_OFF) | RS_INST | RW_WRITE | backlight , io + 2 );
#else
	parport_write_control( pd->port, ((display==1?E1_ON:E2_ON) | RS_INST| RW_WRITE) | backlight );
	udelay( 250 );
	parport_write_data( pd->port, command );
	parport_write_control( pd->port, ((display==1?E1_OFF:E2_OFF) | RS_INST | RW_WRITE) | backlight );
	udelay( 250 );
#endif // CONFIG_PARPORT

#ifdef LOW_VOLTAGE
	/* LOW_VOLTAGE displays operate much slower */
	switch (command)
	{
		case 0x01:
		case 0x02:
		case 0x03:
		case 0x38:
			mdelay( 20 );
			break;
		default:
			udelay( 160 );
			break;
	}
#else
	/* Minimum time to wait after most commands is 39ns except Clear Display
	 * and Return Home which is 1.53ms, we never use Return Home.
	 */
	if( command <= 0x03 )
		mdelay( 1.6 );
	else
		udelay( 40 );
#endif // LOW_VOLTAGE
}

/* Send character data to the display, i.e. writing to DDRAM */
static void writeData( unsigned char data )
{
	/* check and see if we really need to write anything */
	if( state[ row ][ column ] != data )
	{
		state[ row ][ column ] = data;
		/* set the cursor position if need be.
		 * Special case for 16x1 displays, They are treated as two
		 * 8 charcter lines side by side, and dont scroll along to
		 * the second line automaticly.
		 */
		if( disp_row != row || disp_column != column ||
				( disp_rows == 1 && disp_cols == 16 && column == 8 ) )
		{
			if( NUM_CONTROLLERS == 1 )
			{
				/* Some transation done here so 4 line displays work */
				writeCommand( ( row>=2?(row-2)*0x40:row*0x40 ) |
						( row>=2?column+disp_cols:column ) |
						0x80, 1 );
			} else {
				writeCommand( row * 0x40 | column | 0x80, row<2?1:2 );
			}
			disp_row = row;
			disp_column = column;
		}

		/* The E cycle time for a write should be at least 500 ns,
		 * and the minimum E pulse width should be 230 ns.
		 */
#if !CONFIG_PARPORT && !CONFIG_PARPORT_MODULE
		outb(  (row<2||NUM_CONTROLLERS==1?E1_ON:E2_ON) | RS_DATA | RW_WRITE | backlight, io + 2 );
		udelay( 250 );
		outb( ( unsigned char ) data, io );
		outb(  (row<2||NUM_CONTROLLERS==1?E1_OFF:E2_OFF) | RS_DATA | RW_WRITE | backlight, io + 2 );
#else
		parport_write_control( pd->port,(row<2||NUM_CONTROLLERS==1?E1_ON:E2_ON) | RS_DATA | RW_WRITE | backlight );
		udelay( 250 );
		parport_write_data( pd->port, ( unsigned char ) data );
		parport_write_control( pd->port,(row<2||NUM_CONTROLLERS==1?E1_OFF:E2_OFF) | RS_DATA | RW_WRITE | backlight );
#endif // CONFIG_PARPORT
		udelay( 250 );
		/* Time to wait after write to RAM is 43ns */
		udelay( 43 );
		disp_column++;
	}
	if ( column < disp_cols - 1 )
		column++;
	else if ( wrap && column == disp_cols - 1 && row < disp_rows - 1 )
	{
		column = 0;
		row++;
	}
}

/* Write an entire (5x8) character to the CGRAM,
 * takes the CGRAM index, and a char[ 8 ] binary bitmap.
 */
static void writeCGRAM( int index, unsigned char pixels[] )
{
	int i;

	/* Move address pointer to index in CGRAM */
	writeCommand( 0x40 + ( 8 * index ), 1 );
	if( NUM_CONTROLLERS == 2 )
		writeCommand( 0x40 + ( 8 * index ), 2 );
	for( i = 0; i < 8; i++ )
	{
		/* Write a line of a character. The E cycle time for
		 * a write should be at least 500 ns, and the minimum
		 * E pulse width should be 230 ns.
		 */
#if !CONFIG_PARPORT && !CONFIG_PARPORT_MODULE
		outb (E1_ON | RS_DATA | RW_WRITE | backlight, io + 2);
		udelay( 250 );
		outb( ( unsigned char ) pixels[ i ], io );
		outb( E1_OFF | RS_DATA | RW_WRITE | backlight, io + 2 );

		if( NUM_CONTROLLERS == 2 )
		{
			/* Write to controller 2 */
			outb( E2_ON | RS_DATA | RW_WRITE | backlight, io + 2 );
			udelay( 250 );
			outb( ( unsigned char ) pixels[ i ], io );
			outb( E2_OFF | RS_DATA | RW_WRITE | backlight, io + 2 );
		}
#else
		parport_write_control( pd->port, E1_ON | RS_DATA | RW_WRITE | backlight );
		udelay( 250 );
		parport_write_data( pd->port, ( unsigned char ) pixels[ i ] );
		parport_write_control( pd->port, E1_OFF | RS_DATA | RW_WRITE | backlight );
		if( NUM_CONTROLLERS == 2 )
		{
			/* Write to controller 2 */
			parport_write_control( pd->port, E2_ON | RS_DATA | RW_WRITE | backlight );
			udelay( 250 );
			parport_write_data( pd->port, ( unsigned char ) pixels[ i ] );
			parport_write_control( pd->port, E2_OFF | RS_DATA | RW_WRITE | backlight );
		}
#endif // CONFIG_PARPORT
		udelay( 250 );
		/* Time to wait after write to RAM is 43 ns */
		udelay( 45 );
	}
	row = column = -1;
}

static void initDisplay()
{

	/* initialize state array */
	memset( state, ' ', sizeof( state ) );
	/* initialize controller 1 */
	writeCommand( 0x30, 1 );
	writeCommand( 0x30, 1 );
	writeCommand( 0x38, 1 );    // Function set
	writeCommand( 0x0c, 1 );    // Display on
	writeCommand( 0x01, 1 );    // Clear display
	writeCommand( 0x06, 1 );    // Entry mode set
	writeCommand( 0x02, 1 );	// Cursor home

	if( NUM_CONTROLLERS == 2 )
	{
		/* initialize controller 2 */
		writeCommand( 0x30, 2 );
		writeCommand( 0x30, 2 );
		writeCommand( 0x38, 2 );    // Function set
		writeCommand( 0x0c, 2 );    // Display on
		writeCommand( 0x01, 2 );    // Clear display
		writeCommand( 0x06, 2 );    // Entry mode set
		writeCommand( 0x02, 2 );	// Cursor home
	}
	/* Set the CGRAM to default values */
	writeCGRAM( 0, cg0 );
	writeCGRAM( 1, cg1 );
	writeCGRAM( 2, cg2 );
	writeCGRAM( 3, cg3 );
	writeCGRAM( 4, cg4 );
	writeCGRAM( 5, cg5 );
	writeCGRAM( 6, cg6 );
	writeCGRAM( 7, cg7 );
	init_charmap();
}

static void handleInput( unsigned char input )
{
	static int cgram_index = 0;
	static int cgram_row_count;
	static unsigned char cgram_pixels[ 8 ];
	static unsigned char char_map_old;
	static int input_state = NORMAL; 			// the current state of the input handler
	int i;
	int j;
	int temp;

	if ( input_state == NORMAL )
	{
		switch ( input )
		{
			case 0x08: 	// Backspace
				if ( column > 0 )
				{
					column--;
					writeData( ' ' );
					column--;
				}
				break;
			case 0x09: 	// Tabstop
				column = ( ( ( column + 1 ) / TABSTOP ) * TABSTOP ) + TABSTOP - 1;
				break;
			case 0x0a: 	// Newline
				if ( row < disp_rows - 1 )
					row++;
				else
				{
					/* scroll up */
					temp = column;
					for ( i = 0; i < disp_rows - 1; i++ )
					{
						row = i;
						for( j = 0; j < disp_cols; j++ )
						{
							column = j;
							writeData( state[ i + 1 ][ j ] );
						}
					}
					row = disp_rows - 1;
					column = 0;
					for ( i = 0; i < disp_cols; i++ )
					{
						writeData( ' ' );
					}
					column = temp;
				}
				/* Since many have trouble grasping the \r\n concept... */
				column = 0;
				break;
			case 0x0d: 	// Carrage return
				column = 0;
				break;
			case 0x1b: 	// esc ie. start of escape sequence
				input_state = ESC_;
				break;
			default:
				/* The character is looked up in the */
				writeData( charmap[ input ] );
		}
	}
	else if ( input_state == ESC_ )
	{
		input_state = ESC;
	}
	else if ( input_state == ESC )
	{
		if( input <= '7' && input >= '0' )
		{
			/* Chararacter from CGRAM */
			writeData( input - 0x30 );
		} else {
			switch ( input )
			{
				case 'A': 		// Cursor up
					if ( row > 0 )
						row--;
					break;
				case 'B': 		// Cursor down
					if ( row < disp_rows - 1 )
						row++;
					break;
				case 'C': 		// Cursor Right
					if ( column < disp_cols - 1 )
						column++;
					break;
				case 'D': 		// Cursor Left
					if ( column > 0 )
						column--;
					break;
				case 'H': 		// Cursor home
					row = 0;
					column = 0;
					break;
				case 'J': 		// Clear screen, cursor doesn't move
					memset( state, ' ', sizeof( state ) );
					writeCommand( 0x01, 1 );
					if( NUM_CONTROLLERS == 2 )
						writeCommand( 0x01, 2 );
					break;
				case 'K': 		// Erase to end of line, cursor doesn't move
					temp = column;
					for ( i = column; i < disp_cols; i++ )
						writeData( ' ' );
					column = temp;
					break;
				case 'M':		// Charater mapping
					input_state = CHAR_MAP_OLD;
					break;
				case 'Y': 		// Direct cursor access
					input_state = DCA_Y;
					break;
				case 'R':		// CGRAM select
					input_state = CGRAM_SELECT;
					break;
				case 'V':		// Linewrap on
					wrap = 1;
					break;
				case 'W':		// Linewrap off
					wrap = 0;
					break;
               case 'b':       // Toggle backlight
                   backlight = ( backlight == BL_OFF ? BL_ON : BL_OFF );
                   break;
				default:
					printk( "LCD: unrecognized escape sequence: %#x ('%c')\n", input, input );
			}
		}
		if ( input_state != DCA_Y &&
				input_state != CGRAM_SELECT &&
				input_state != CHAR_MAP_OLD )
		{
			input_state = NORMAL;
		}
	}
	else if ( input_state == DCA_Y )
	{
		if ( input - 0x1f < disp_rows )
			row = input - 0x1f;
		else
		{
			printk( "LCD: tried to set cursor to off screen location\n" );
			row = disp_rows - 1;
		}
		input_state = DCA_X;
	}
	else if ( input_state == DCA_X )
	{
		if ( input - 0x1f < disp_cols )
			column = input - 0x1f;
		else
		{
			printk( "LCD: tried to set cursor to off screen location\n" );
			column = disp_cols - 1;
		}
		input_state = NORMAL;
	}
	else if ( input_state == CGRAM_SELECT )
	{
		if( input > '7' || input < '0' )
		{
			printk( "LCD: Bad CGRAM index %c\n", input );
			input_state = NORMAL;
		} else {
			cgram_index = input - 0x30;
			cgram_row_count = 0;
			input_state = CGRAM_GENERATE;
		}
	}
	else if( input_state == CGRAM_GENERATE )
	{
		cgram_pixels[ cgram_row_count++ ] = input;
		if( cgram_row_count == 8 )
		{
			writeCGRAM( cgram_index, cgram_pixels );
			input_state = NORMAL;
		}
	}
	else if( input_state == CHAR_MAP_OLD )
	{
		char_map_old = input;
		input_state = CHAR_MAP_NEW;
	}
	else if( input_state == CHAR_MAP_NEW )
	{
		charmap[ char_map_old ] = input;
		input_state = NORMAL;
	}
}

/* Handle device open */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,0)
static int lcd_open( struct inode *minode, struct file *file )
#else
int lcd_open( struct inode *minode, struct file *mfile )
#endif
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	if ( MOD_IN_USE )
		return -EBUSY;

	MOD_INC_USE_COUNT;
#else
// FIXME : 2.6.x inc use count
#endif
	return ( 0 );
}

/* Handle device file close */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,0)
static void lcd_release( struct inode *minode, struct file *mfile )
#else
int lcd_release( struct inode *minode, struct file *mfile )
#endif
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	MOD_DEC_USE_COUNT;
#else
// FIXME : 2.6.x dec use count
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,0)
	return 0;
#endif
}

/* Handle write to device file */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,0)
int lcd_write_byte( struct inode *minode, struct file *mfile, const char *gdata, int length )
#else
ssize_t lcd_write_byte( struct file *inode, const char *gdata, size_t length, loff_t *off_what )
#endif
{
	int i;
	for ( i = 0; i < length; i++ )
		handleInput( gdata[ i ] );
	return ( length );
}

/* Handle read from device file */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,0)
int lcd_read_byte( struct inode *inode, struct file *mfile, char *data, int length )
#else
ssize_t lcd_read_byte( struct file *inode, char *udata, size_t length, loff_t *loff_what )
#endif
{
	return ( EACCES );
}

/* Handle read from proc file */
int readProcFile( char *buffer, char **start, off_t offset, int size, int *eof, void *data )

{
	char *temp = buffer;
	int i, j;

	/* Print module configuration */
	temp += sprintf( temp, "I/O base:        %#x\n"
			"Display rows:    %d\n"
			"Display columns: %d\n"
			"Linewrap:        %s\n\n",
			io, disp_rows, disp_cols,
			wrap?"On":"Off" );

	/* Print display state */
	temp += sprintf( temp, "+" );
	for( i = 0; i < disp_cols; i++ )
		temp += sprintf( temp, "-" );
	temp += sprintf( temp, "+\n" );

	for( i = 0; i < disp_rows; i++ )
	{
		temp += sprintf( temp, "|" );
		for( j = 0; j < disp_cols ; j++ )
			temp += sprintf( temp, "%c", (state[ i ][ j ] < 10)?'':state[ i ][ j ] );
		temp += sprintf( temp, "|\n" );
	}
	
	temp += sprintf( temp, "+" );
	for( i = 0; i < disp_cols; i++ )
		temp += sprintf( temp, "-" );
	temp += sprintf( temp, "+\n" );
	return temp - buffer;
}

/* Module cleanup */
void cleanup_module()
{
	writeCommand( 0x01, 1 );
	if( NUM_CONTROLLERS == 2 )
		writeCommand( 0x01, 2 );
#if !CONFIG_PARPORT && !CONFIG_PARPORT_MODULE
	release_region( io, 3 );
#else
 	parport_release( pd );
	parport_unregister_device( pd );
#endif // CONFIG_PARPORT
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
	remove_proc_entry( "lcd", 0 );
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)

	if ( unregister_chrdev( LCD_MAJOR, "LCD panel" ) )
		printk( "LCD: module cleanup failed\n" );
	else
		printk( "LCD: module cleanup OK\n" );
#else
  dev_t dev = MKDEV(LCD_MAJOR,0);
  printk ("LCD: deleting char device %d\n", dev);
  // these functions return void so no error checking possible
  unregister_chrdev_region (dev, 1);
  cdev_del (my_cdev );
#endif
}

#if CONFIG_PARPORT || CONFIG_PARPORT_MODULE
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
/* parport_find_base is missing in 2.2.x kernels
 * Thanks Nick Metcalfe (krane@alphalink.com.au)
 */
struct parport *parport_find_base( int io )
{
	struct parport *list;
	list = parport_enumerate();
	while( list != NULL && list->base != io )
		list = list->next;
	return list;
}
#endif
#endif

/* Module initilisation */
int init_module()
{

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,0)
	static struct file_operations lcd_fops =
	    { NULL, lcd_read_byte, lcd_write_byte, NULL, NULL, NULL, NULL, lcd_open, lcd_release, };
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	static struct file_operations lcd_fops =
	    { NULL, lcd_read_byte, lcd_write_byte, NULL, NULL, NULL, NULL, lcd_open, NULL, lcd_release, };
#elif  LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	static struct file_operations lcd_fops =
	    { NULL, NULL, lcd_read_byte, lcd_write_byte, NULL, NULL, NULL, NULL, lcd_open, NULL, lcd_release, };
#else 
        static struct file_operations lcd_fops =
            { NULL, NULL, lcd_read_byte, NULL, lcd_write_byte, NULL, NULL, NULL, NULL, NULL, lcd_open, NULL, lcd_release, };
#endif

#if !CONFIG_PARPORT && !CONFIG_PARPORT_MODULE
	if( check_region( io, 3 ) )
	{
		printk( "LCD: check_region failed, io = %#x\n", io );
		return -EACCES;
	}
	else if( request_region( io, 3, "LCD panel" ) )
	{
		printk( "LCD: request_region failed, io = %#x\n", io );
		return -EACCES;
	} else {
#else
	static struct parport *pp;

	pp = parport_find_base( io );
	if( !pp )
	{
		printk( "LCD: no parport found at %#x\n", io );
		return -EACCES;
	}

	if( ( pd = parport_register_device( pp, "lcdmod", NULL, NULL,
				NULL, PARPORT_DEV_EXCL, NULL ) ) == NULL )
	{
		printk( "LCD: parport busy\n" );
		return -EBUSY;
	}
	if( parport_claim_or_block( pd ) < 0 )
	{
		printk( "LCD: couldn't claim port\n" );
		parport_release( pd );
		parport_unregister_device( pd );
		return -EAGAIN;
	} else {
#endif	// CONFIG_PARPORT
		/* Make sure user didn't pass silly numbers, MAX_DISP_???? are just
		 * arbitary numbers and can be increased if need be.
		 */
		disp_rows = disp_rows<=MAX_DISP_ROWS?disp_rows:MAX_DISP_ROWS;
		disp_cols = disp_cols<=MAX_DISP_COLS?disp_cols:MAX_DISP_COLS;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
        if ( register_chrdev( LCD_MAJOR, "LCD panel", &lcd_fops ) )
                {
                        printk( "LCD: Error! Unable to bind to major %d\n", LCD_MAJOR );
#if !CONFIG_PARPORT && !CONFIG_PARPORT_MODULE
                        release_region( io, 3 );
#else
                        parport_release( pd );
                        parport_unregister_device( pd );
#endif // CONFIG_PARPORT
                        return -1;
                }
#else 
		// device number
		dev_t dev = MKDEV(LCD_MAJOR, 0);
		if ( register_chrdev_region(dev, 1, "LCD panel") ) 
		{
		  printk ("LCD: char device register failed\n"); 
		  return -1; 
		} 
		else 
		{
		  printk ("LCD: char device registered\n"); 
		}
		// fill cdev structure
		my_cdev = cdev_alloc();
		my_cdev->owner = THIS_MODULE;
		my_cdev->ops = &lcd_fops;
		kobject_set_name(&my_cdev->kobj, "LCD panel");

		// register it
		if ( cdev_add(my_cdev, dev, 1) ) 
		{ 
		  printk ("LCD char device add failed\n");
#if !CONFIG_PARPORT && !CONFIG_PARPORT_MODULE
		    release_region( io, 3 );
#else
		    parport_release( pd );
		    parport_unregister_device( pd );
#endif // CONFIG_PARPORT
		  return -1; 
		} 
		else 
		  printk ("LCD: char device added\n"); 
		
#endif // LINUX_VERSION_CODE

		initDisplay();
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
		if( !create_proc_read_entry( "lcd", 0, 0, readProcFile, NULL ) )
		{
			printk( KERN_ERR "LCD: Error! Can't create /proc/lcd\n" );
			return -ENOMEM;
		}
#endif
		printk( "LCD: init OK, iobase: %#x, rows: %d, columns: %d\n", io, disp_rows, disp_cols );
	}
	return 0;
}
